function [ext_img,ref_img,se] = imextract(varargin)
%This function takes a reference image or stack, dilates, and uses it as a 
%mask on the series of images or stacks, thus extracting out only those
%features that exist in the reference image.
%Synatax:   [ext_img,dil_img,se] = imextract(ref_img,target_img,'se',20,'se_type','ball','threshold','30'); 
%Input:     ref_img = the image or image stack that will be used as the
%               reference for image extraction.  After image extraction
%               only the elements in  the location of this image will be
%               present in all of the other input images.
%           target_img = the image or image stacks that will be provode the
%               base data for image extracton.  target_img can be a cell
%               array of images or image stacks.  NOTE: target_img must be
%               the same dimensions as the ref_img.
%           'se' = is the parameters for the structuring element created by
%               the strel function: 
%                 SE = strel(shape, parameters)
%                 SE = strel('arbitrary', nhood)
%                 SE = strel('arbitrary', nhood, h)
%                 SE = strel('ball', r, h, n)
%                 SE = strel('diamond', r)
%                 SE = strel('disk', r, n)
%                 SE = strel('line', len, deg)
%                 SE = strel('octagon', r)
%                 SE = strel('pair', offset)
%                 SE = strel('periodicline', p, v)
%                 SE = strel('rectangle', mn)
%                 SE = strel('square', w)
%           'se_type' = is the structuring element type used.  Default =
%               ball.
%           'threshold' = is the cut off of pixel values used to create the
%               the mask image.  Thus, all pixels => than threshold will be
%               extracted.  Threshold is defined as a percentage.  Default
%               = 0.5.  Value must be between [0 1].
%           'inv' = invert the selection, so that above the threshold is
%               rejected. Default = 0. 0 = off.
%           'imgnum' = select the two image number you want to use for
%               threshold selection.  Select one image with the highest
%               contrast and one with the lowest contrast in the stack.
%           NOTE: if  you want to enter a parameter for the structuring
%           element you must enter all of the parameters for that
%           operation.
%           SPECIAL: If r is set to 0, then there is no dilation of the
%           reference image. If threshold = [], threshold =
%           graythresh(img), which is Otsu's method for calculating the
%           threshold

[ref_img,target_img,se,stks_ch_names,channel_paths] = parse(varargin);
%lets extract out the interesting bits.
mkdir(channel_paths,'extracted');      %create output directory
if size(target_img,2)~=1        %multiple stacks
    for i = 1:size(target_img,2)
        ext_img{1,i} = im2single(target_img{1,i}).*ref_img;
        mkdir([channel_paths,'extracted'],stks_ch_names{1,i}(1:end-4));
        stk2tiff(ext_img{1,i},stks_ch_names{1,i},[channel_paths,'extracted\',stks_ch_names{1,i}(1:end-4),'\']);
    end
else        %one stack only
    ext_img = im2single(target_img{1,1}).*ref_img;
    mkdir([channel_paths,'extracted'],stks_ch_names{1,1}(1:end-4));
    stk2tiff(ext_img,stks_ch_names{1,1},[channel_paths,'extracted\',stks_ch_names{1,1}(1:end-4),'\']);
end
%output ref_img
mkdir([channel_paths,'extracted'],'ref_img');
stk2tiff(ref_img,'ref_img',[channel_paths,'extracted\ref_img\']);
        

%--------------------------------------------------------------------------
%subfunction to parse the inputs.
function [ref_img,target_img,se,stks_ch_names,channel_paths] = parse(input)

se_type = 'ball';   %default ball structuring element.
threshold = 0.5;     %default = 50%.
ref_img = [];       %set images empty for now
target_img = [];    
r = 3;              %dilation on.
imgnum = [];        %the selected image number
inv = 0;            %invert off.

%Parse the input
if ~isempty(input)
    for i = 1:2:size(input,2)
        if ischar(input{1,i});
            switch input{1,i}
                case 'threshold'
                    threshold = input{1,i+1};
                case 'imgnum'
                    imgnum = input{1,i+1};
                case 'inv'
                    inv = input{1,i+1};
                case 'se_type'
                    if ischar(input{1,i+1})
                        se_type = input{1,i+1};
                    else
                        warning(['Your entered structuring element type is not recognized, reverting to defalut type.']);
                    end
                case 'nhood'
                    nhood = input{1,i+1};
                case 'r'    %radius
                    r = input{1,i+1};
                case 'h'    %height
                    h = input{1,i+1};
                case 'n'
%                     When N is greater than 0, the ball-shaped structuring element is approximated by a sequence of N nonflat, 
%                     line-shaped structuring elements. When N equals 0, no approximation is used, and the structuring element 
%                     members consist of all pixels whose centers are no greater than R away from the origin. The corresponding 
%                     height values are determined from the formula of the ellipsoid specified by R and H. If N is not specified, 
%                     the default value is 8.
                    n = input{1,i+1};
                case 'len'    %length
                    len = input{1,i+1};
                case 'deg'
                    deg = input{1,i+1};
                case 'offset'
                    offset = input{1,i+1};
                case 'p'    %radius
                    p = input{1,i+1};
                case 'v'    %height
                    v = input{1,i+1};
                case 'mn'    %sides of the rectangle
                    mn = input{1,i+1};
                case 'w'    %sides of the square
                    w = input{1,i+1};
                case 'ref_img'
                    ref_img = input{1,i+1};
                case 'target_img'
                    target_img = input{1,i+1};
                otherwise
                    warning(['Your input ',input{1,i},' is not recognized.']);
            end
        else
            error(['The parameters you entered is incorrect.  Please check help.']);
        end
    end
end
%create the structuring element
switch se_type
    case 'ball'     %note: if you want to use a parameter for the structuring element, you must enter all of the parameters.
        try
            se = strel('ball',r,h,n);   %check to see if parameters are defined
        catch
            se = strel('ball',3,3);     %if not default, n = 8 in this case.
        end
    case 'arbitrary'     
        try
            se = strel('arbitrary',nhood,h);   %check to see if parameters are defined
        catch
            se = strel('arbitrary',5,5);     %if not default.
        end
    case 'diamond'
        try
            se = strel('diamond',r);   %check to see if parameters are defined
        catch
            se = strel('diamond',5);     %if not default.
        end
    case 'disk'
        try
            se = strel('disk',r,n);   %check to see if parameters are defined
        catch
            se = strel('disk',5);     %if not. default n = 4
        end
    case 'line'
        try
            se = strel('line',len,deg);   %check to see if parameters are defined
        catch
            se = strel('line',5,0);     %if not default.
        end
    case 'octagon'
        try
            se = strel('octagon',r);   %check to see if parameters are defined
        catch
            se = strel('octagon',5);     %if not default.
        end
    case 'pair'
        try
            se = strel('pair',offset);   %check to see if parameters are defined
        catch
            se = strel('pair',[5 5]);     %if not default.
        end
    case 'periodicline'
        try
            se = strel('periodicline',p,v);   %check to see if parameters are defined
        catch
            se = strel('periodicline',5,[5 5]);     %if not default.
        end
    case 'rectangle'
        try
            se = strel('rectangle',mn);   %check to see if parameters are defined
        catch
            se = strel('rectangle',[5 5]);     %if not default. square!
        end
    case 'square'
        try
            se = strel('square',w);
        catch
            se = strel('square',5);
        end
end
%open the image stacks
if isempty(ref_img)
    prompt_box('title','Open Reference Image','prompt1','Select image stack ',...
        'prompt2','as the reference image','position','center');
    pause(0.25);
    [ref_img] = stack_gui(1);  %use cell output mode.
    ref_img = imnorm(ref_img{1,1});  %don't need cell for this, normalize for better performance.
end
%dilate the ref_img, unless r=0.
if r~=0
    ref_img = imdilate(ref_img,se);     %3d dilate a bit for best 3d performance.
end
%OK now lets play with the image for best selection.
%se2 = strel('disk',3);    %dilate a little bit
if size(ref_img,3)==1   %there is only 1 image
    imgnum = [1 1];     %the one
    hi_lo = 0;      %don't need to show two images
else
    if isempty(imgnum)  %if the image number is not specified.
        %pick the image to use for display.  We want the one with the least
        %contrast
        for k = 1:size(ref_img,3)   %step through the stack
            img_std(k) = mean(std(single(ref_img(:,:,k))));     %grab the standard deviation of the image
            img_int(k) = sum(sum(single(ref_img(:,:,k))));      %grab the total intensity of the image
        end
        img_std = img_std/mean(img_std);    %normalize
        img_int = img_int/mean(img_int);    %ditto
        img_score = img_std+img_int;    %create an image score
        imgnum(1) = find(img_score==min(img_score));  %We want the lowest score
        imgnum(2) = find(img_score==max(img_score));  %and the highest score
    end
    hi_lo = 1;  %two images please
end
threshold = play_thresh(ref_img(:,:,imgnum(1)),ref_img(:,:,imgnum(2)),threshold,se,hi_lo,r);   %pick your threshold
ref_bw = false(size(ref_img));      %preallocate
for j = 1:size(ref_img,3)
%     if r~=0
%         ref_bw(:,:,j) = imfill(im2bw(imdilate(ref_img(:,:,j),se),threshold),'holes');       %apply to entire ref_img stack.
%     else    %no dilation
        ref_bw(:,:,j) = im2bw(ref_img(:,:,j),threshold);
%     end
end
ref_img = ref_bw;   %reasign back to the original image stack.
%open the target images
if isempty(target_img)
    prompt_box('title','Open Target Images','prompt1','Select image stack or stacks ',...
        'prompt2','for feature extraction','position','center');
    pause(0.25);
    [target_img,img_range,stks_ch_names,channel_paths] = stack_gui(1);  %use cell output mode.
end
%--------------------------------------------------------------------------
function [thresh] = play_thresh(img,img2,thresh,se,hi_lo,r)
%this function lets the user select a threshold interactively
%if inital thresh is not given, default to 0.1
if nargin<3
    thresh = 0.1;
    se = strel('disk',0);   %no dilation
    hi_lo = 1;
    r = 0;
end
if nargin<4
    se = strel('disk',0);   %no dilation
    hi_lo = 1;
    r = 0;
end
ithresh = 0;        %initiate
warning('off')
%show original
if hi_lo    %show second, because it is different
    h3 = figure('Name','Original High Contrast','NumberTitle','off');
    imshow(img2);
    h2 = figure('Name','Processed High Contrast','NumberTitle','off');
end
h1 = figure('Name','Original Low Contrast','NumberTitle','off');
imshow(img);
h = figure('Name','Processed Low Contrast','NumberTitle','off');    %create image
while ithresh==0
%     if r~=0 %no dilation
%         img_bw1 = imfill(im2bw(imdilate(img,se),thresh),'holes');
%     else
        img_bw1 = im2bw(img,thresh);
%     end
    if hi_lo    %show second, because it is different
%         if r~=0
%             img_bw2 = imfill(im2bw(imdilate(img2,se),thresh),'holes');
%         else
            img_bw2 = im2bw(img2,thresh);
%         end
        figure(h2);     %select figure
        imshow(img_bw2);
    end
    figure(h);     %select figure
    %hold
    imshow(img_bw1);
    ithresh = yes_no_box('title','Is this thresholded image OK?','caption1','Select yes is you want to move on.',...
        'caption2','Note: try an image that is background subtracted','position','west');
    pause(0.25);
    if ithresh==0
        thresh = str2double(response_box('title','Enter your new threshold','input',num2str(thresh),'position','center'));
        pause(0.25);
    end
end
close(h1);
close(h)
if hi_lo
    close(h2)
    close(h3)
end
warning('on')